
#include "sram.h"
#include "p33Fxxxx.h"

static unsigned short current_mem_addr, pending_mem_addr;
static unsigned char mem_addr_pending;

void __attribute__((__interrupt__,no_auto_psv)) _SPI1Interrupt(void) {
  IFS0bits.SPI1IF = 0;

  mem_addr_pending = 2;

  SPI1STATbits.SPIEN = 0;
}

static void send_spi_word(unsigned short word) {
  // send 16 bits of data
  SPI1STATbits.SPIEN = 1;
  SPI1BUF = word;
  // bring RCK/CS low
  LATAbits.LATA1 = 0;
}

void sram_prepare_addr(unsigned short addr) {
  mem_addr_pending = 1;
  pending_mem_addr = addr;
  send_spi_word(addr);
}

static void finish_pending_mem_change() {
  LATAbits.LATA1 = 1;
  current_mem_addr = pending_mem_addr;
  mem_addr_pending = 0;
}

void sram_change_addr(unsigned short addr) {
  if( mem_addr_pending == 1 && pending_mem_addr == addr ) {
    while( SPI1STATbits.SPIEN )
      ;
  }
  if( mem_addr_pending == 2 )
    finish_pending_mem_change();
  if( current_mem_addr != addr ) {
    sram_prepare_addr(addr);
    while( SPI1STATbits.SPIEN )
      ;
    finish_pending_mem_change();
  }
}

void sram_init() {
  // RA1 = RCK/CS (digital output, default high)
  TRISAbits.TRISA1 = 0;
  LATAbits.LATA1 = 1;
  AD1PCFGLbits.PCFG2 = 1;

  // RB0/RP0 = SDO (digital output, default high)
  // connect to SPI1 Data Output, 0x07
  AD1PCFGLbits.PCFG1 = 1;
  TRISBbits.TRISB0 = 0;
  LATBbits.LATB0 = 0;
  RPOR0bits.RP0R = 0x07;

  // RB1/RP1 = SCK (digital output, default high)
  // connect to SPI1 Clock Output, 0x08
  AD1PCFGLbits.PCFG3 = 1;
  TRISBbits.TRISB1 = 0;
  LATBbits.LATB1 = 0;
  RPOR0bits.RP1R = 0x08;

  // set up SPI1
#ifdef SLOW_CLOCK
  SPI1CON1bits.PPRE = 0x3;
  SPI1CON1bits.SPRE = 0x6;
#else
  SPI1CON1bits.PPRE = 0x2;
  SPI1CON1bits.SPRE = 0x7;
#endif
  SPI1CON1bits.MSTEN = 1;
  SPI1CON1bits.CKP = 1;
  SPI1CON1bits.MODE16 = 1;
  IFS0bits.SPI1IF = 0;
  IEC0bits.SPI1IE = 1;

  // RB3 = memory data bus (digital input)
  AD1PCFGLbits.PCFG5 = 1;
  TRISBbits.TRISB3 = 1;

  // RA3 = memory OE-bar (digital output, default high)
  TRISAbits.TRISA3 = 0;
  LATAbits.LATA3 = 1;

  // RB4 = memory data bus (digital input)
  TRISBbits.TRISB4 = 1;

  // RA4 = memory WR-bar (digital output, default high)
  TRISAbits.TRISA4 = 0;
  LATAbits.LATA4 = 1;

  // RB5-10 = memory data bus (digital input)
  TRISBbits.TRISB5 = 1;
  TRISBbits.TRISB6 = 1;
  TRISBbits.TRISB7 = 1;
  TRISBbits.TRISB8 = 1;
  TRISBbits.TRISB9 = 1;
  TRISBbits.TRISB10 = 1;

  // RB11-13 = memory address bus (digital output, default low)
  TRISBbits.TRISB11 = 0;
  LATBbits.LATB11 = 0;
  AD1PCFGLbits.PCFG12 = 1;
  TRISBbits.TRISB12 = 0;
  LATBbits.LATB12 = 0;
  AD1PCFGLbits.PCFG11 = 1;
  TRISBbits.TRISB13 = 0;
  LATBbits.LATB13 = 0;

  current_mem_addr = 0xFFFF;
  sram_change_addr(0);
}

void sram_select_byte(unsigned char which) {
  LATB = (LATB&~((1<<11)|(1<<12)|(1<<13))) | ((unsigned short)which<<11);
}

//#define _DEBUG_

void sram_write_byte(unsigned char addr, unsigned char data) {
  // data bus is RB3-RB10
  LATB = (LATB&~0x3FF8)|((unsigned short)data<<3)|((unsigned short)addr<<11);
  TRISB &= ~0x7F8;
#ifdef _DEBUG_
  asm("nop");
  asm("nop");
#endif
  LATAbits.LATA4 = 0;
#ifndef SLOW_CLOCK
  asm("nop");
#endif
  asm("nop");
  TRISB |=  0x7F8;
  LATAbits.LATA4 = 1;
}

unsigned char sram_read_byte(unsigned char addr) {
  unsigned char ret;
  sram_select_byte(addr);
  LATAbits.LATA3 = 0;
#ifndef SLOW_CLOCK
  asm("nop");
  asm("nop");
#endif
  asm("nop");
  asm("nop");
#ifdef _DEBUG_
  asm("nop");
  asm("nop");
  asm("nop");
  asm("nop");
  asm("nop");
#endif
  ret = PORTB>>3;
  LATAbits.LATA3 = 1;
  return ret;
}
